<?php

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\DashboardController;
use App\Http\Controllers\ReimbursementController;
use App\Http\Controllers\StaffController;
use App\Http\Controllers\StockController;
use App\Http\Controllers\TaskController;
use App\Http\Controllers\AttendanceController;
use App\Http\Controllers\PayrollController;
use App\Http\Controllers\PerformanceController;
use App\Http\Controllers\LeaveTypeController;
use App\Http\Controllers\LeaveRequestController;
use App\Http\Controllers\DepartmentController;
use Illuminate\Support\Facades\Auth;

Route::get('/', function () {
    return redirect()->route('login');
});

Auth::routes(['register' => false]);

Route::middleware(['auth'])->group(function () {
    Route::get('/dashboard', [DashboardController::class, 'index'])->name('dashboard');
    Route::get('/home', [DashboardController::class, 'index'])->name('home');
    
    // Admin only routes
    Route::middleware(['role:Admin'])->group(function () {
        Route::get('/staff/sample', [StaffController::class, 'downloadSample'])->name('staff.sample');
        Route::post('/staff/import', [StaffController::class, 'import'])->name('staff.import');
        Route::resource('staff', StaffController::class);
        
        // Department Management
        Route::resource('departments', DepartmentController::class)->only(['index', 'store', 'destroy']);

        // Payrolls resource moved to shared routes
        Route::get('/payrolls/report', [PayrollController::class, 'report'])->name('payrolls.report');
        Route::get('/payrolls/bulk/{batchId}', [PayrollController::class, 'showBulk'])->name('payrolls.bulk.show');
        Route::delete('/payrolls/bulk/{batchId}', [PayrollController::class, 'destroyBatch'])->name('payrolls.bulk.destroy');
        Route::post('/payrolls/{payroll}/approve', [PayrollController::class, 'approve'])->name('payrolls.approve');
        Route::post('/payrolls/{payroll}/mark-paid', [PayrollController::class, 'markPaid'])->name('payrolls.markPaid');
        
        // Performance Analytics
        Route::get('/performance', [PerformanceController::class, 'index'])->name('performance.index');
        Route::get('/performance/export', [PerformanceController::class, 'export'])->name('performance.export');
        
        // Stock Management
        Route::get('/stock/upload', [StockController::class, 'uploadForm'])->name('stock.upload.form');
        Route::post('/stock/upload', [StockController::class, 'upload'])->name('stock.upload');

        // Leave Management (Admin)
        Route::resource('leaves/types', LeaveTypeController::class, ['as' => 'leaves']);

        // Reimbursement Management
        Route::get('/reimbursements', [ReimbursementController::class, 'index'])->name('reimbursements.index');
        Route::patch('/reimbursements/{reimbursement}/status', [ReimbursementController::class, 'updateStatus'])->name('reimbursements.updateStatus');
    });
    
    // Routes accessible by both Admin and Staff
    Route::resource('payrolls', PayrollController::class);
    Route::get('/payrolls/{payroll}/download', [PayrollController::class, 'downloadPdf'])->name('payrolls.downloadPdf');
    Route::resource('leaves/requests', LeaveRequestController::class, ['as' => 'leaves']);
    Route::resource('tasks', TaskController::class);
    Route::get('/my-attendance', [AttendanceController::class, 'staffAttendance'])->name('attendance.staff');
    Route::post('/attendance/mark', [AttendanceController::class, 'markAttendance'])->name('attendance.mark');
    
    // Stock Management (accessible by both Admin and Staff)
    Route::get('/stock', [StockController::class, 'index'])->name('stock.index');
    Route::post('/stock', [StockController::class, 'index'])->name('stock.index.post');
    
    // Attendance routes (accessible by both Admin and Staff based on policy)
    Route::get('/attendance/all', [AttendanceController::class, 'index'])->name('attendance.index');
Route::resource('attendance', AttendanceController::class)->except(['index']);
    Route::get('/attendance/map', [AttendanceController::class, 'map'])->name('attendance.map');
    Route::get('/attendance/report', [AttendanceController::class, 'report'])->name('attendance.report');
    Route::post('/attendance/quick-update', [AttendanceController::class, 'quickUpdate'])->name('attendance.quick-update');
    Route::post('/attendance/checkin', [AttendanceController::class, 'checkin'])->name('attendance.checkin');
    Route::post('/attendance/checkout', [AttendanceController::class, 'checkout'])->name('attendance.checkout');
    
    // Task specific routes
    Route::get('/tasks/{task}/reassign', [TaskController::class, 'showReassignForm'])->name('tasks.reassign.form');
    Route::post('/tasks/{task}/reassign', [TaskController::class, 'reassign'])->name('tasks.reassign');
    Route::post('/tasks/{task}/complete', [TaskController::class, 'complete'])->name('tasks.complete');
    Route::get('/tasks/{task}/assignment-history', [TaskController::class, 'assignmentHistory'])->name('tasks.assignment-history');
    Route::get('/tasks/{task}/create-subtask', [TaskController::class, 'createSubtask'])->name('tasks.create-subtask');
});



// API routes for AJAX requests
Route::middleware(['auth'])->prefix('api')->group(function () {
    Route::get('/departments/{department}/staff', function ($departmentId) {
        return \App\Models\Department::find($departmentId)->staff;
    });
    Route::get('/departments/{department}/users', [TaskController::class, 'getUsersByDepartment'])->name('api.departments.users');
});
