<?php

namespace App\Services;

class ExifService
{
    /**
     * Extract GPS coordinates from image EXIF data
     *
     * @param string $imagePath
     * @return array|null
     */
    public static function extractGpsCoordinates($imagePath)
    {
        try {
            // Check if file exists and is readable
            if (!file_exists($imagePath) || !is_readable($imagePath)) {
                return null;
            }

            // Read EXIF data
            $exifData = @exif_read_data($imagePath);
            
            if (!$exifData || !isset($exifData['GPS'])) {
                return null;
            }

            $gps = $exifData['GPS'];

            // Check if required GPS data exists
            if (!isset($gps['GPSLatitude']) || !isset($gps['GPSLongitude']) ||
                !isset($gps['GPSLatitudeRef']) || !isset($gps['GPSLongitudeRef'])) {
                return null;
            }

            // Convert GPS coordinates to decimal degrees
            $latitude = self::convertGpsToDecimal($gps['GPSLatitude'], $gps['GPSLatitudeRef']);
            $longitude = self::convertGpsToDecimal($gps['GPSLongitude'], $gps['GPSLongitudeRef']);

            if ($latitude === null || $longitude === null) {
                return null;
            }

            return [
                'latitude' => $latitude,
                'longitude' => $longitude,
                'altitude' => isset($gps['GPSAltitude']) ? self::convertGpsAltitude($gps['GPSAltitude'], $gps['GPSAltitudeRef'] ?? '0') : null,
                'timestamp' => self::extractGpsTimestamp($gps),
                'raw_exif' => $gps
            ];

        } catch (\Exception $e) {
            //\Log::error('EXIF GPS extraction failed: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Convert GPS coordinates from degrees/minutes/seconds to decimal degrees
     *
     * @param array $coordinate
     * @param string $hemisphere
     * @return float|null
     */
    private static function convertGpsToDecimal($coordinate, $hemisphere)
    {
        try {
            if (!is_array($coordinate) || count($coordinate) < 3) {
                return null;
            }

            // Convert fractions to decimal
            $degrees = self::fractionToDecimal($coordinate[0]);
            $minutes = self::fractionToDecimal($coordinate[1]);
            $seconds = self::fractionToDecimal($coordinate[2]);

            if ($degrees === null || $minutes === null || $seconds === null) {
                return null;
            }

            // Convert to decimal degrees
            $decimal = $degrees + ($minutes / 60) + ($seconds / 3600);

            // Apply hemisphere (negative for South/West)
            if (in_array($hemisphere, ['S', 'W'])) {
                $decimal = -$decimal;
            }

            return round($decimal, 8);

        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * Convert GPS altitude to decimal meters
     *
     * @param string $altitude
     * @param string $ref
     * @return float|null
     */
    private static function convertGpsAltitude($altitude, $ref)
    {
        try {
            $decimal = self::fractionToDecimal($altitude);
            if ($decimal === null) {
                return null;
            }

            // Apply reference (1 = below sea level)
            if ($ref === '1') {
                $decimal = -$decimal;
            }

            return round($decimal, 2);

        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * Extract GPS timestamp from EXIF data
     *
     * @param array $gps
     * @return string|null
     */
    private static function extractGpsTimestamp($gps)
    {
        try {
            if (!isset($gps['GPSDateStamp']) || !isset($gps['GPSTimeStamp'])) {
                return null;
            }

            $date = $gps['GPSDateStamp'];
            $time = $gps['GPSTimeStamp'];

            if (!is_array($time) || count($time) < 3) {
                return null;
            }

            $hours = self::fractionToDecimal($time[0]);
            $minutes = self::fractionToDecimal($time[1]);
            $seconds = self::fractionToDecimal($time[2]);

            if ($hours === null || $minutes === null || $seconds === null) {
                return null;
            }

            return sprintf('%s %02d:%02d:%02d UTC', $date, $hours, $minutes, $seconds);

        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * Convert EXIF fraction string to decimal
     *
     * @param string|int|float $fraction
     * @return float|null
     */
    private static function fractionToDecimal($fraction)
    {
        try {
            if (is_numeric($fraction)) {
                return (float) $fraction;
            }

            if (is_string($fraction) && strpos($fraction, '/') !== false) {
                $parts = explode('/', $fraction);
                if (count($parts) === 2 && is_numeric($parts[0]) && is_numeric($parts[1]) && $parts[1] != 0) {
                    return (float) $parts[0] / (float) $parts[1];
                }
            }

            return null;

        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * Extract all relevant EXIF metadata from image
     *
     * @param string $imagePath
     * @return array
     */
    public static function extractAllMetadata($imagePath)
    {
        try {
            if (!file_exists($imagePath) || !is_readable($imagePath)) {
                return [];
            }

            $exifData = @exif_read_data($imagePath);
            
            if (!$exifData) {
                return [];
            }

            $metadata = [
                'camera_make' => $exifData['Make'] ?? null,
                'camera_model' => $exifData['Model'] ?? null,
                'datetime_original' => $exifData['DateTimeOriginal'] ?? null,
                'datetime_digitized' => $exifData['DateTimeDigitized'] ?? null,
                'orientation' => $exifData['Orientation'] ?? null,
                'x_resolution' => $exifData['XResolution'] ?? null,
                'y_resolution' => $exifData['YResolution'] ?? null,
                'software' => $exifData['Software'] ?? null,
            ];

            // Add GPS data if available
            $gpsData = self::extractGpsCoordinates($imagePath);
            if ($gpsData) {
                $metadata['gps_data'] = $gpsData;
            }

            return array_filter($metadata, function($value) {
                return $value !== null;
            });

        } catch (\Exception $e) {
            //\Log::error('EXIF metadata extraction failed: ' . $e->getMessage());
            return [];
        }
    }
}