<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Support\Facades\Auth;

class Task extends Model
{
    protected $fillable = [
        'title',
        'description',
        'created_by',
        'current_department_id',
        'current_assigned_to',
        'status',
        'deadline',
        'priority',
        'is_overdue',
        'parent_id',
    ];

    protected $casts = [
        'deadline' => 'datetime',
        'is_overdue' => 'boolean',
    ];

    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function currentDepartment(): BelongsTo
    {
        return $this->belongsTo(Department::class, 'current_department_id');
    }

    public function currentAssignee(): BelongsTo
    {
        return $this->belongsTo(User::class, 'current_assigned_to');
    }

    public function logs(): HasMany
    {
        return $this->hasMany(TaskLog::class)->orderBy('created_at');
    }

    /**
     * Get all assignments for this task.
     */
    public function assignments(): HasMany
    {
        return $this->hasMany(TaskAssignment::class)->orderBy('assigned_at', 'desc');
    }

    /**
     * Get the current active assignment.
     */
    public function currentAssignment(): HasOne
    {
        return $this->hasOne(TaskAssignment::class)->where('status', 'active')->latest('assigned_at');
    }

    /**
     * Get assignment history (all assignments except current).
     */
    public function assignmentHistory(): HasMany
    {
        return $this->hasMany(TaskAssignment::class)
                    ->whereIn('status', ['reassigned', 'completed'])
                    ->orderBy('assigned_at', 'desc');
    }

    /**
     * Get the parent task.
     */
    public function parent(): BelongsTo
    {
        return $this->belongsTo(Task::class, 'parent_id');
    }

    /**
     * Get all subtasks.
     */
    public function subtasks(): HasMany
    {
        return $this->hasMany(Task::class, 'parent_id')->orderBy('created_at');
    }

    /**
     * Get only active (non-completed) subtasks.
     */
    public function activeSubtasks(): HasMany
    {
        return $this->subtasks()->whereNotIn('status', ['completed']);
    }

    /**
     * Assign task to a user.
     */
    public function assignTo(User $user, User $assignedBy, $notes = null)
    {
        // Mark current assignment as reassigned if exists
        $this->assignments()->where('status', 'active')->update(['status' => 'reassigned']);
        
        // Create new assignment
        $assignment = $this->assignments()->create([
            'assigned_by' => $assignedBy->id,
            'assigned_to' => $user->id,
            'department_id' => $user->staffProfile->department_id ?? null,
            'status' => 'active',
            'notes' => $notes,
            'assigned_at' => now(),
        ]);
        
        // Update current assignment fields in tasks table
        $this->update([
            'current_assigned_to' => $user->id,
            'current_department_id' => $user->staffProfile->department_id ?? $this->current_department_id,
        ]);
        
        return $assignment;
    }

    /**
     * Mark task as completed by Super Admin.
     */
    public function markCompleted(User $completedBy)
    {
        // Mark current assignment as completed
        $this->assignments()->where('status', 'active')->update([
            'status' => 'completed',
            'completed_at' => now(),
        ]);
        
        // Update task status with completed_at timestamp
        $this->update([
            'status' => 'completed',
            'completed_at' => now()
        ]);
        
        // Log the completion
        $this->logs()->create([
            'action' => 'completed',
            'user_id' => $completedBy->id,
            'details' => 'Task marked as completed by Super Admin',
        ]);
    }

    /**
     * Automatically set completed_at when status is changed to completed
     */
    public function setStatusAttribute($value)
    {
        $this->attributes['status'] = $value;
        
        if ($value === 'completed' && !$this->completed_at) {
            $this->attributes['completed_at'] = now();
        } elseif ($value !== 'completed') {
            $this->attributes['completed_at'] = null;
        }
    }

    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    public function scopeInProgress($query)
    {
        return $query->where('status', 'in_progress');
    }

    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    /**
     * Check if task is overdue
     */
    public function isOverdue(): bool
    {
        if (!$this->deadline || $this->status === 'completed') {
            return false;
        }
        return now()->gt($this->deadline);
    }

    /**
     * Get days until deadline
     */
    public function daysUntilDeadline(): ?int
    {
        if (!$this->deadline) {
            return null;
        }
        return now()->diffInDays($this->deadline, false);
    }

    /**
     * Auto-escalate priority based on deadline proximity
     */
    public function escalatePriorityByDeadline(): void
    {
        if (!$this->deadline || $this->status === 'completed') {
            return;
        }

        $daysUntil = $this->daysUntilDeadline();
        
        if ($daysUntil < 0) {
            // Overdue - set to urgent
            $this->update(['priority' => 'urgent', 'is_overdue' => true]);
        } elseif ($daysUntil <= 1) {
            // 1 day or less - set to high if not already urgent
            if ($this->priority !== 'urgent') {
                $this->update(['priority' => 'high']);
            }
        } elseif ($daysUntil <= 3) {
            // 3 days or less - set to medium if currently low
            if ($this->priority === 'low') {
                $this->update(['priority' => 'medium']);
            }
        }
    }

    /**
     * Scope for overdue tasks
     */
    public function scopeOverdue($query)
    {
        return $query->where('is_overdue', true)
                    ->orWhere(function($q) {
                        $q->whereNotNull('deadline')
                          ->where('deadline', '<', now())
                          ->where('status', '!=', 'completed');
                    });
    }

    /**
     * Scope for tasks by priority
     */
    public function scopeByPriority($query, $priority)
    {
        return $query->where('priority', $priority);
    }

    /**
     * Scope for tasks due soon (within specified days)
     */
    public function scopeDueSoon($query, $days = 3)
    {
        return $query->whereNotNull('deadline')
                    ->where('deadline', '>=', now())
                    ->where('deadline', '<=', now()->addDays($days))
                    ->where('status', '!=', 'completed');
    }

    /**
     * Scope for parent tasks only (no parent_id)
     */
    public function scopeParentTasks($query)
    {
        return $query->whereNull('parent_id');
    }

    /**
     * Scope for subtasks only (has parent_id)
     */
    public function scopeSubtasks($query)
    {
        return $query->whereNotNull('parent_id');
    }

    /**
     * Check if this task is a parent task
     */
    public function isParentTask(): bool
    {
        return $this->subtasks()->exists();
    }

    /**
     * Check if this task is a subtask
     */
    public function isSubtask(): bool
    {
        return !is_null($this->parent_id);
    }

    /**
     * Get subtask completion percentage
     */
    public function getSubtaskCompletionPercentage(): int
    {
        $totalSubtasks = $this->subtasks()->count();
        if ($totalSubtasks === 0) {
            return 0;
        }
        
        $completedSubtasks = $this->subtasks()->where('status', 'completed')->count();
        return round(($completedSubtasks / $totalSubtasks) * 100);
    }

    /**
     * Auto-complete parent task when all subtasks are completed
     */
    public function checkAndCompleteParent()
    {
        if ($this->parent && $this->parent->activeSubtasks()->count() === 0) {
            $this->parent->update(['status' => 'completed']);
            $user = Auth::user();
            // Log the auto-completion
            $this->parent->logs()->create([
                'action' => 'auto_completed',
                'user_id' => $user->id,
                'details' => 'Task auto-completed as all subtasks are finished',
            ]);
        }
    }
}
