<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Notification extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'user_id',
        'task_id',
        'title',
        'message',
        'type',
        'notification_category',
        'priority_level',
        'is_persistent',
        'is_dismissible',
        'read_at',
        'expires_at',
        'metadata',
    ];

    protected $casts = [
        'is_persistent' => 'boolean',
        'is_dismissible' => 'boolean',
        'read_at' => 'datetime',
        'expires_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'deleted_at' => 'datetime',
        'metadata' => 'array',
    ];

    /**
     * Get the user that owns the notification.
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the task associated with the notification.
     */
    public function task()
    {
        return $this->belongsTo(Task::class);
    }

    /**
     * Scope to get unread notifications
     */
    public function scopeUnread($query)
    {
        return $query->whereNull('read_at');
    }

    /**
     * Scope to get read notifications
     */
    public function scopeRead($query)
    {
        return $query->whereNotNull('read_at');
    }

    /**
     * Scope to get persistent notifications
     */
    public function scopePersistent($query)
    {
        return $query->where('is_persistent', true);
    }

    /**
     * Scope to get non-persistent notifications
     */
    public function scopeNonPersistent($query)
    {
        return $query->where('is_persistent', false);
    }

    /**
     * Scope to get task persistent notifications
     */
    public function scopeTaskPersistent($query)
    {
        return $query->where('notification_category', 'task_persistent')
                    ->where('is_persistent', true);
    }

    /**
     * Scope to get dismissible notifications
     */
    public function scopeDismissible($query)
    {
        return $query->where('is_dismissible', true);
    }

    /**
     * Scope to get non-dismissible notifications
     */
    public function scopeNonDismissible($query)
    {
        return $query->where('is_dismissible', false);
    }

    /**
     * Scope to get notifications by priority level
     */
    public function scopeByPriority($query, $priority)
    {
        return $query->where('priority_level', $priority);
    }

    /**
     * Scope to get notifications by category
     */
    public function scopeByCategory($query, $category)
    {
        return $query->where('notification_category', $category);
    }

    /**
     * Scope to get active notifications (not expired)
     */
    public function scopeActive($query)
    {
        return $query->where(function($q) {
            $q->whereNull('expires_at')
              ->orWhere('expires_at', '>', now());
        });
    }

    /**
     * Scope to get expired notifications
     */
    public function scopeExpired($query)
    {
        return $query->whereNotNull('expires_at')
                    ->where('expires_at', '<=', now());
    }

    /**
     * Scope to get notifications for a specific user
     */
    public function scopeForUser($query, $userId)
    {
        return $query->where(function($q) use ($userId) {
            $q->where('user_id', $userId)
              ->orWhere('user_id', null); // Global notifications
        });
    }

    /**
     * Check if notification is read
     */
    public function isRead()
    {
        return !is_null($this->read_at);
    }

    /**
     * Check if notification is expired
     */
    public function isExpired()
    {
        return $this->expires_at && $this->expires_at->isPast();
    }

    /**
     * Check if notification is active (not expired)
     */
    public function isActive()
    {
        return !$this->isExpired();
    }

    /**
     * Mark notification as read
     */
    public function markAsRead()
    {
        $this->update(['read_at' => now()]);
    }

    /**
     * Mark notification as unread
     */
    public function markAsUnread()
    {
        $this->update(['read_at' => null]);
    }

    /**
     * Check if notification is dismissible
     */
    public function isDismissible()
    {
        return $this->is_dismissible;
    }

    /**
     * Check if notification is task persistent
     */
    public function isTaskPersistent()
    {
        return $this->notification_category === 'task_persistent' && $this->is_persistent;
    }

    /**
     * Get the notification priority color
     */
    public function getPriorityColorAttribute()
    {
        $colors = [
            'low' => 'green',
            'medium' => 'blue',
            'high' => 'orange',
            'urgent' => 'red',
        ];

        return $colors[$this->priority_level] ?? 'gray';
    }

    /**
     * Get the notification type badge color
     */
    public function getTypeBadgeColorAttribute()
    {
        $colors = [
            'info' => 'blue',
            'warning' => 'yellow',
            'success' => 'green',
            'error' => 'red',
            'announcement' => 'purple',
        ];

        return $colors[$this->type] ?? 'gray';
    }

    /**
     * Get formatted created at date
     */
    public function getFormattedCreatedAtAttribute()
    {
        return $this->created_at->format('M d, Y h:i A');
    }

    /**
     * Get time ago format
     */
    public function getTimeAgoAttribute()
    {
        return $this->created_at->diffForHumans();
    }
}