<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;
use Illuminate\Http\Request;

class Attendance extends Model
{
    protected $fillable = [
        'user_id',
        'date',
        'check_in',
        'check_out',
        'hours_worked',
        'status',
        'notes',
        'selfie_path',
        'selfie_metadata',
        'latitude',
        'longitude',
        'location_address',
        'device_info',
        'marked_at',
        'attendance_type',
        'check_out_time',
        'check_out_selfie_path',
        'check_out_latitude',
        'check_out_longitude',
        'check_out_location_address',
    ];

    protected $casts = [
        'date' => 'date',
        'check_in' => 'datetime:H:i',
        'check_out' => 'datetime:H:i',
        'hours_worked' => 'decimal:2',
        'device_info' => 'array',
        'selfie_metadata' => 'array',
        'marked_at' => 'datetime',
        'latitude' => 'decimal:8',
        'longitude' => 'decimal:8',
        'check_out_time' => 'datetime',
        'check_out_latitude' => 'decimal:8',
        'check_out_longitude' => 'decimal:8',
    ];

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function scopeForMonth($query, $year, $month)
    {
        return $query->whereYear('date', $year)->whereMonth('date', $month);
    }

    public function scopePresent($query)
    {
        return $query->where('status', 'present');
    }

    public function scopeAbsent($query)
    {
        return $query->where('status', 'absent');
    }

    public function scopeLate($query)
    {
        return $query->where('status', 'late');
    }

    public function scopeHalfDay($query)
    {
        return $query->where('status', 'half_day');
    }

    /**
     * Check if the date is a working day (Monday to Saturday)
     */
    public static function isWorkingDay($date)
    {
        $dayOfWeek = Carbon::parse($date)->dayOfWeek;
        return $dayOfWeek >= 1 && $dayOfWeek <= 6; // Monday (1) to Saturday (6)
    }

    /**
     * Get working days between two dates
     */
    public static function getWorkingDaysBetween($startDate, $endDate)
    {
        $start = Carbon::parse($startDate);
        $end = Carbon::parse($endDate);
        $workingDays = 0;

        while ($start->lte($end)) {
            if (self::isWorkingDay($start)) {
                $workingDays++;
            }
            $start->addDay();
        }

        return $workingDays;
    }

    public function apiIndex(Request $request)
    {
        // Similar logic to index() but return JSON
        $attendances = Attendance::with(['user'])->paginate(15);
        return response()->json($attendances);
    }

    /**
     * Check if attendance has selfie
     */
    public function hasSelfie()
    {
        if (empty($this->selfie_path)) {
            return false;
        }
        
        // Check in main domain folder instead of Laravel app public folder
        $mainDomainPath = base_path('../attendence_nsoindia_com/' . $this->selfie_path);
        return file_exists($mainDomainPath);
    }

    /**
     * Get selfie URL
     */
    public function getSelfieUrl()
    {
        if ($this->hasSelfie()) {
            return asset($this->selfie_path);
        }
        return null;
    }

    /**
     * Check if attendance has location data
     */
    public function hasLocation()
    {
        return !empty($this->latitude) && !empty($this->longitude);
    }

    /**
     * Get formatted location
     */
    public function getFormattedLocation()
    {
        if ($this->hasLocation()) {
            return $this->latitude . ', ' . $this->longitude;
        }
        return null;
    }

    /**
     * Check if attendance has check-out selfie
     */
    public function hasCheckOutSelfie()
    {
        if (empty($this->check_out_selfie_path)) {
            return false;
        }
        
        // Check in main domain folder instead of Laravel app public folder
        $mainDomainPath = base_path('../attendence_nsoindia_com/' . $this->check_out_selfie_path);
        return file_exists($mainDomainPath);
    }

    /**
     * Get check-out selfie URL
     */
    public function getCheckOutSelfieUrl()
    {
        if ($this->hasCheckOutSelfie()) {
            return asset($this->check_out_selfie_path);
        }
        return null;
    }

    /**
     * Check if attendance has check-out location data
     */
    public function hasCheckOutLocation()
    {
        return !empty($this->check_out_latitude) && !empty($this->check_out_longitude);
    }

    /**
     * Get formatted check-out location
     */
    public function getFormattedCheckOutLocation()
    {
        if ($this->hasCheckOutLocation()) {
            return $this->check_out_latitude . ', ' . $this->check_out_longitude;
        }
        return null;
    }

    /**
     * Scope for manual attendance entries
     */
    public function scopeManual($query)
    {
        return $query->where('attendance_type', 'manual');
    }

    /**
     * Scope for admin attendance entries
     */
    public function scopeAdmin($query)
    {
        return $query->where('attendance_type', 'admin');
    }

    public function apiStore(Request $request)
    {
        // Similar logic to store() but return JSON
        $validated = $request->validate([
            // validation rules
        ]);
        
        $attendance = Attendance::create($validated);
        return response()->json($attendance, 201);
    }
}
