<?php

namespace App\Http\Controllers;

use App\Models\Task;
use App\Models\TaskAssignment;
use App\Models\TaskLog;
use App\Models\Department;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class TaskController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        
        // Start building the query
        $query = Task::with(['creator', 'currentDepartment', 'currentAssignee', 'subtasks', 'parent']);
        
        // Base query based on user role
        if (!$user->hasRole('Admin')) {
            $query->where('current_assigned_to', $user->id);
        }
        
        // Employee name search
        if ($request->filled('employee_search')) {
            $searchTerm = $request->employee_search;
            $query->whereHas('currentAssignee', function($q) use ($searchTerm) {
                $q->where('name', 'LIKE', "%{$searchTerm}%")
                  ->orWhere('email', 'LIKE', "%{$searchTerm}%");
            });
        }
        
        // Department filter
        if ($request->filled('department')) {
            $query->where('current_department_id', $request->department);
        }
        
        // Status filter
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }
        
        // Priority filter
        if ($request->filled('priority')) {
            $query->where('priority', $request->priority);
        }
        
        // Date range filters
        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }
        
        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }
        
        // Task type filter
        if ($request->filled('task_type')) {
            switch ($request->task_type) {
                case 'parent':
                    $query->whereHas('subtasks');
                    break;
                case 'subtask':
                    $query->whereNotNull('parent_id');
                    break;
                case 'main':
                    $query->whereNull('parent_id');
                    break;
            }
        }
        
        // Overdue/deadline status filter
        if ($request->filled('overdue')) {
            switch ($request->overdue) {
                case 'overdue':
                    $query->where('deadline', '<', now())
                          ->where('status', '!=', 'completed');
                    break;
                case 'due_soon':
                    $query->whereBetween('deadline', [now(), now()->addDays(2)])
                          ->where('status', '!=', 'completed');
                    break;
                case 'no_deadline':
                    $query->whereNull('deadline');
                    break;
            }
        }
        
        $tasks = $query->latest()->paginate(15)->withQueryString();
        
        return view('tasks.index', compact('tasks'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request)
    {
        $this->authorize('create', Task::class);
        
        $departments = Department::all();
        $users = User::role('Staff')->with('staffProfile.department')->get();
        // Get all ongoing tasks (both parent tasks and main tasks) that can serve as parent tasks
        $parentTasks = Task::whereNull('parent_id')
                          ->whereIn('status', ['pending', 'in_progress'])
                          ->orderBy('title')
                          ->get(['id', 'title']);
        $parentId = $request->get('parent_id');
        
        return view('tasks.create', compact('departments', 'users', 'parentTasks', 'parentId'));
    }

    /**
     * Show the form for creating a subtask for a specific parent task.
     */
    public function createSubtask(Task $task)
    {
        $this->authorize('create', Task::class);
        
        // Ensure the task can be a parent (not already a subtask)
        if ($task->parent_id) {
            return redirect()->route('tasks.index')
                           ->with('error', 'Cannot create subtask for a task that is already a subtask.');
        }
        
        $departments = Department::all();
        $users = User::role('Staff')->with('staffProfile.department')->get();
        
        return view('tasks.create-subtask', compact('task', 'departments', 'users'));
    }

    /**
     * Get users by department for AJAX requests
     */
    public function getUsersByDepartment($departmentId)
    {
        $users = User::role('Staff')
            ->whereHas('staffProfile', function($query) use ($departmentId) {
                $query->where('department_id', $departmentId);
            })
            ->with('staffProfile')
            ->get(['id', 'name']);
        
        return response()->json($users);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $this->authorize('create', Task::class);
        
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'current_department_id' => 'required|exists:departments,id',
            'current_assigned_to' => 'nullable|exists:users,id',
            'deadline' => 'nullable|date|after:now',
            'priority' => 'nullable|in:low,medium,high,urgent',
            'notes' => 'nullable|string',
            'parent_id' => 'nullable|exists:tasks,id',
        ]);
        
        DB::transaction(function () use ($validated) {
            $task = Task::create([
                'title' => $validated['title'],
                'description' => $validated['description'],
                'created_by' => Auth::id(),
                'current_department_id' => $validated['current_department_id'],
                'current_assigned_to' => $validated['current_assigned_to'],
                'deadline' => $validated['deadline'] ?? null,
                'priority' => $validated['priority'] ?? 'medium',
                'status' => 'pending',
                'parent_id' => $validated['parent_id'] ?? null,
            ]);
            
            // Create initial assignment if user is specified
            if ($validated['current_assigned_to']) {
                $assignedUser = User::find($validated['current_assigned_to']);
                $task->assignTo($assignedUser, Auth::user(), $validated['notes'] ?? 'Initial assignment');
            }
            
            // Log task creation
            TaskLog::create([
                'task_id' => $task->id,
                'action' => 'created',
                'performed_by' => Auth::id(),
                'to_department_id' => $validated['current_department_id'],
                'to_user_id' => $validated['current_assigned_to'],
                'notes' => 'Task created',
            ]);
        });
        
        return redirect()->route('tasks.index')
                        ->with('success', 'Task created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Task $task)
    {
        $task->load([
            'creator', 
            'currentDepartment', 
            'currentAssignee', 
            'logs.performer', 
            'logs.fromDepartment', 
            'logs.toDepartment', 
            'logs.fromUser', 
            'logs.toUser',
            'assignments.assignedBy',
            'assignments.assignedTo',
            'assignments.department',
            'currentAssignment.assignedBy',
            'currentAssignment.assignedTo',
            'parent',
            'subtasks.currentAssignee',
            'subtasks.currentDepartment'
        ]);
        
        return view('tasks.show', compact('task'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Task $task)
    {
        $this->authorize('update', $task);
        
        $departments = Department::all();
        $users = User::role('Staff')->get();
        
        return view('tasks.edit', compact('task', 'departments', 'users'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Task $task)
    {
        $this->authorize('update', $task);
        
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'current_department_id' => 'required|exists:departments,id',
            'current_assigned_to' => 'nullable|exists:users,id',
            'status' => 'required|in:pending,in_progress,completed',
        ]);
        
        $oldDepartment = $task->current_department_id;
        $oldAssignee = $task->current_assigned_to;
        $oldStatus = $task->status;
        
        $task->update($validated);
        
        // Log changes
        $changes = [];
        if ($oldDepartment != $validated['current_department_id']) {
            $changes[] = 'department changed';
        }
        if ($oldAssignee != $validated['current_assigned_to']) {
            $changes[] = 'assignee changed';
        }
        if ($oldStatus != $validated['status']) {
            $changes[] = 'status changed to ' . $validated['status'];
        }
        
        if (!empty($changes)) {
            TaskLog::create([
                'task_id' => $task->id,
                'action' => 'updated',
                'performed_by' => Auth::id(),
                'from_department_id' => $oldDepartment,
                'to_department_id' => $validated['current_department_id'],
                'from_user_id' => $oldAssignee,
                'to_user_id' => $validated['current_assigned_to'],
                'notes' => 'Task updated: ' . implode(', ', $changes),
            ]);
        }
        
        // Check if parent task should be auto-completed when subtask is completed
        if ($oldStatus != 'completed' && $validated['status'] == 'completed' && $task->parent_id) {
            $task->checkAndCompleteParent();
        }
        
        return redirect()->route('tasks.show', $task)
                        ->with('success', 'Task updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Task $task)
    {
        $this->authorize('delete', $task);
        
        $task->delete();
        
        return redirect()->route('tasks.index')
                        ->with('success', 'Task deleted successfully.');
    }
    
    /**
     * Show the reassign form
     */
    public function showReassignForm(Task $task)
    {
        $this->authorize('reassign', $task);
        
        return view('tasks.reassign', compact('task'));
    }
    
    /**
     * Reassign task to another user
     */
    public function reassign(Request $request, Task $task)
    {
        $this->authorize('reassign', $task);
        
        $validated = $request->validate([
            'assigned_to' => 'required|exists:users,id',
            'notes' => 'nullable|string|max:500',
        ]);
        
        $assignedUser = User::find($validated['assigned_to']);
        
        DB::transaction(function () use ($task, $assignedUser, $validated) {
            // Create new assignment using the model method
            $task->assignTo($assignedUser, Auth::user(), $validated['notes'] ?? 'Task reassigned');
            
            // Update task status to in_progress if it was pending
            if ($task->status === 'pending') {
                $task->update(['status' => 'in_progress']);
            }
            
            // Log the reassignment
            TaskLog::create([
                'task_id' => $task->id,
                'action' => 'reassigned',
                'performed_by' => Auth::id(),
                'to_user_id' => $assignedUser->id,
                'notes' => 'Task reassigned: ' . ($validated['notes'] ?? 'No notes provided'),
            ]);
        });
        
        return redirect()->route('tasks.show', $task)->with('success', 'Task reassigned successfully.');
    }
    
    /**
     * Mark task as completed (Super Admin only)
     */
    public function complete(Task $task)
    {
        $this->authorize('complete', $task);
        
        DB::transaction(function () use ($task) {
            $task->markCompleted(Auth::user());
        });
        
        return back()->with('success', 'Task marked as completed.');
    }
    
    /**
     * View assignment history for a task
     */
    public function assignmentHistory(Task $task)
    {
        $this->authorize('viewAssignmentHistory', $task);
        
        $task->load([
            'assignments.assignedBy',
            'assignments.assignedTo',
            'assignments.department'
        ]);
        
        return view('tasks.assignment-history', compact('task'));
    }

    // API Methods for Mobile App
    
    /**
     * API: Get tasks
     */
    public function apiIndex(Request $request)
    {
        $user = $request->user();
        
        if ($user->hasRole('Admin')) {
            $query = Task::with([
                'creator', 
                'currentDepartment', 
                'currentAssignee',
                'subtasks.currentAssignee',
                'subtasks.currentDepartment'
            ]);
            
            // Filter by status if provided
            if ($request->filled('status')) {
                $query->where('status', $request->status);
            }
            
            // Filter by department if provided
            if ($request->filled('department_id')) {
                $query->where('current_department_id', $request->department_id);
            }
            
            $tasks = $query->latest()->paginate(15);
        } else {
            // Staff can only see their assigned tasks
            $query = Task::where('current_assigned_to', $user->id)
                        ->with([
                            'creator', 
                            'currentDepartment',
                            'subtasks.currentAssignee',
                            'subtasks.currentDepartment'
                        ]);
            
            if ($request->filled('status')) {
                $query->where('status', $request->status);
            }
            
            $tasks = $query->latest()->paginate(15);
        }
        
        return response()->json($tasks);
    }
    
    /**
     * API: Create task
     */
    public function apiStore(Request $request)
    {
        $user = $request->user();
        
        // Only admin can create tasks
        if (!$user->hasRole('Admin')) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }
        
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'current_department_id' => 'required|exists:departments,id',
            'current_assigned_to' => 'nullable|exists:users,id',
        ]);
        
        $validated['created_by'] = $user->id;
        $validated['status'] = 'pending';
        
        $task = Task::create($validated);
        
        // Log the task creation
        TaskLog::create([
            'task_id' => $task->id,
            'performed_by' => $user->id,
            'action' => 'created',
            'notes' => 'Task created',
        ]);
        
        return response()->json([
            'message' => 'Task created successfully.',
            'task' => $task->load(['creator', 'currentDepartment', 'currentAssignee'])
        ], 201);
    }
    
    /**
     * API: Get specific task
     */
    public function apiShow(Request $request, $id)
    {
        $user = $request->user();
        $task = Task::with([
            'creator', 
            'currentDepartment', 
            'currentAssignee', 
            'subtasks.currentAssignee',
            'subtasks.currentDepartment',
            'logs.performer',
            'logs.fromDepartment',
            'logs.toDepartment',
            'logs.fromUser',
            'logs.toUser'
        ])->findOrFail($id);
        
        // Staff can only view their assigned tasks
        if (!$user->hasRole('Admin') && $task->current_assigned_to !== $user->id) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }
        
        return response()->json($task);
    }
    
    /**
     * API: Update task
     */
    public function apiUpdate(Request $request, $id)
    {
        $user = $request->user();
        $task = Task::findOrFail($id);
        
        // Only admin can update tasks
        if (!$user->hasRole('Admin')) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }
        
        $validated = $request->validate([
            'title' => 'sometimes|string|max:255',
            'description' => 'sometimes|string',
            'current_department_id' => 'sometimes|exists:departments,id',
            'current_assigned_to' => 'nullable|exists:users,id',
            'status' => 'sometimes|in:pending,in_progress,completed,cancelled',
        ]);
        
        $oldValues = $task->only(array_keys($validated));
        $task->update($validated);
        
        // Log changes
        foreach ($validated as $field => $newValue) {
            if (isset($oldValues[$field]) && $oldValues[$field] != $newValue) {
                TaskLog::create([
                    'task_id' => $task->id,
                    'performed_by' => $user->id,
                    'action' => 'updated',
                    'notes' => "Updated {$field} from '{$oldValues[$field]}' to '{$newValue}'",
                ]);
            }
        }
        
        return response()->json([
            'message' => 'Task updated successfully.',
            'task' => $task->fresh()->load(['creator', 'currentDepartment', 'currentAssignee'])
        ]);
    }
    
    /**
     * API: Delete task
     */
    public function apiDestroy(Request $request, $id)
    {
        $user = $request->user();
        
        // Only admin can delete tasks
        if (!$user->hasRole('Admin')) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }
        
        $task = Task::findOrFail($id);
        
        // Log the deletion
        TaskLog::create([
            'task_id' => $task->id,
            'performed_by' => $user->id,
            'action' => 'deleted',
            'notes' => 'Task deleted',
        ]);
        
        $task->delete();
        
        return response()->json(['message' => 'Task deleted successfully.']);
    }
    
    /**
     * API: Mark task as complete
     */
    public function apiComplete(Request $request, $id)
    {
        $user = $request->user();
        $task = Task::findOrFail($id);
        
        // Staff can only complete their assigned tasks
        if (!$user->hasRole('Admin') && $task->current_assigned_to !== $user->id) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }
        
        $validated = $request->validate([
            'completion_notes' => 'nullable|string|max:1000',
        ]);
        
        $task->update([
            'status' => 'completed',
        ]);
        
        // Log the completion
        TaskLog::create([
            'task_id' => $task->id,
            'performed_by' => $user->id,
            'action' => 'completed',
            'notes' => 'Task marked as completed' . ($validated['completion_notes'] ? ': ' . $validated['completion_notes'] : ''),
        ]);
        
        return response()->json([
            'message' => 'Task marked as completed.',
            'task' => $task->fresh()->load(['creator', 'currentDepartment', 'currentAssignee'])
        ]);
    }
    
    /**
     * API: Reassign task
     */
    public function apiReassign(Request $request, $id)
    {
        $user = $request->user();
        $task = Task::findOrFail($id);
        
        // Admin or currently assigned user can reassign tasks
        if (!$user->hasRole('Admin') && $task->current_assigned_to !== $user->id) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }
        
        $validated = $request->validate([
            'current_assigned_to' => 'required|exists:users,id',
            'current_department_id' => 'sometimes|exists:departments,id',
            'reassignment_notes' => 'nullable|string|max:1000',
        ]);
        
        $oldAssignee = $task->current_assigned_to;
        $oldDepartment = $task->current_department_id;
        
        $task->update([
            'current_assigned_to' => $validated['current_assigned_to'],
            'current_department_id' => $validated['current_department_id'] ?? $task->current_department_id,
        ]);
        
        // Log the reassignment
        TaskLog::create([
            'task_id' => $task->id,
            'performed_by' => $user->id,
            'action' => 'reassigned',
        ]);
        
        return response()->json([
            'message' => 'Task reassigned successfully.',
            'task' => $task->fresh()->load(['creator', 'currentDepartment', 'currentAssignee'])
        ]);
    }
    
    /**
     * API: Get task assignment history
     */
    public function apiAssignmentHistory(Request $request, $id)
    {
        $user = $request->user();
        $task = Task::findOrFail($id);
        
        // Staff can only view history of their assigned tasks
        if (!$user->hasRole('Admin') && $task->current_assigned_to !== $user->id) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }
        
        $logs = TaskLog::where('task_id', $id)
                      ->with('user')
                      ->latest()
                      ->get();
        
        return response()->json([
            'task' => $task->load(['creator', 'currentDepartment', 'currentAssignee']),
            'logs' => $logs
        ]);
    }

    /**
     * API: Create a subtask
     */
    public function apiCreateSubtask(Request $request, $id)
    {
        $user = $request->user();
        $parentTask = Task::findOrFail($id);
        
        // Admin and Staff can create subtasks
        if (!$user->hasRole('Admin') && !$user->hasRole('Staff')) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }
        
        // Validate that parent task exists and is not already a subtask
        if ($parentTask->parent_id) {
            return response()->json(['error' => 'Cannot create subtask of a subtask'], 400);
        }
        
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'priority' => 'nullable|in:Low,Medium,High',
            'due_date' => 'nullable|date',
            'assigned_to' => 'nullable|exists:users,id',
        ]);
        
        // Create subtask with parent relationship
        $subtask = Task::create([
            'title' => $validated['title'],
            'description' => $validated['description'] ?? '',
            'priority' => $validated['priority'] ?? 'Medium',
            'due_date' => $validated['due_date'] ?? null,
            'parent_id' => $parentTask->id,
            'current_department_id' => $parentTask->current_department_id,
            'current_assigned_to' => $validated['assigned_to'] ?? $parentTask->current_assigned_to,
            'created_by' => $user->id,
            'status' => 'pending',
        ]);
        
        // Log the subtask creation
        TaskLog::create([
            'task_id' => $subtask->id,
            'performed_by' => $user->id,
            'action' => 'created',
            'notes' => "Subtask created under parent task: {$parentTask->title}",
        ]);
        
        return response()->json([
            'message' => 'Subtask created successfully.',
            'subtask' => $subtask->load(['creator', 'currentDepartment', 'currentAssignee', 'parent'])
        ], 201);
    }
}
