<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Stock;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;

class StockController extends Controller
{
    /**
     * Display the stock upload form
     */
    public function uploadForm()
    {
        return view('stock.upload');
    }

    /**
     * Handle CSV file upload and process stock data
     */
    public function upload(Request $request)
    {
        $request->validate([
            'csv_file' => 'required|file|mimes:csv,txt|max:10240', // 10MB max
        ]);

        try {
            $file = $request->file('csv_file');
            $path = $file->store('uploads', 'local');
            $fullPath = Storage::disk('local')->path($path);

            // Process CSV file
            $this->processCsvFile($fullPath);

            // Delete the uploaded file after processing
            Storage::disk('local')->delete($path);

            return redirect()->route('stock.index')->with('success', 'Stock data uploaded successfully!');
        } catch (\Exception $e) {
            return back()->with('error', 'Error uploading file: ' . $e->getMessage());
        }
    }

    /**
     * Process CSV file and insert data into database
     */
    private function processCsvFile($filePath)
    {
        $handle = fopen($filePath, 'r');
        
        if ($handle === false) {
            throw new \Exception('Could not open CSV file');
        }

        // Read header row
        $header = fgetcsv($handle);
        
        // Clear existing stock data (optional - you might want to keep historical data)
        // Stock::truncate();

        $batchSize = 100;
        $batch = [];
        $rowCount = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (count($row) >= 7) { // Ensure we have all required columns
                $batch[] = [
                    'name' => $row[0] ?? '',
                    'product' => $row[1] ?? '',
                    'style' => $row[2] ?? '',
                    'color' => $row[3] ?? '',
                    'size' => $row[4] ?? '',
                    'batch_no' => (int)($row[5] ?? 0),
                    'current_balance_qty' => (int)($row[6] ?? 0),
                    'uploaded_by' => Auth::user()->name ?? 'System',
                    'uploaded_at' => now(),
                    'created_at' => now(),
                    'updated_at' => now(),
                ];

                $rowCount++;

                // Insert in batches for better performance
                if (count($batch) >= $batchSize) {
                    Stock::insert($batch);
                    $batch = [];
                }
            }
        }

        // Insert remaining records
        if (!empty($batch)) {
            Stock::insert($batch);
        }

        fclose($handle);
    }

    /**
     * Display stock data with filters
     */
    public function index(Request $request)
    {
        $query = Stock::query();

        // Apply search filters (real-time search)
        if ($request->filled('search_name')) {
            $query->where('name', 'LIKE', '%' . $request->search_name . '%');
        }

        if ($request->filled('search_product')) {
            $query->where('product', 'LIKE', '%' . $request->search_product . '%');
        }

        if ($request->filled('search_style')) {
            $query->where('style', 'LIKE', '%' . $request->search_style . '%');
        }

        // Apply dropdown filters
        if ($request->filled('name')) {
            $query->filterByName($request->name);
        }

        if ($request->filled('product')) {
            $query->filterByProduct($request->product);
        }

        if ($request->filled('style')) {
            $query->filterByStyle($request->style);
        }

        if ($request->filled('color')) {
            $query->filterByColor($request->color);
        }

        if ($request->filled('size')) {
            $query->filterBySize($request->size);
        }

        if ($request->filled('batch_no')) {
            $query->filterByBatchNo($request->batch_no);
        }

        // Get paginated results
        $stocks = $query->orderBy('created_at', 'desc')->paginate(50);

        // Get unique values for filter dropdowns
        $filterData = [
            'names' => Stock::getUniqueNames(),
            'products' => Stock::getUniqueProducts(),
            'styles' => Stock::getUniqueStyles(),
            'colors' => Stock::getUniqueColors(),
            'sizes' => Stock::getUniqueSizes(),
            'batchNos' => Stock::getUniqueBatchNos(),
        ];

        // Check if this is an AJAX request
        if ($request->ajax() || $request->wantsJson()) {
            return view('stock.partials.table', compact('stocks'))->render();
        }

        return view('stock.index', compact('stocks', 'filterData'));
    }

    /**
     * API endpoint to get stock data for mobile app
     */
    public function apiIndex(Request $request)
    {
        $query = Stock::query();

        // Apply filters
        if ($request->filled('name')) {
            $query->filterByName($request->name);
        }

        if ($request->filled('product')) {
            $query->filterByProduct($request->product);
        }

        if ($request->filled('style')) {
            $query->filterByStyle($request->style);
        }

        if ($request->filled('color')) {
            $query->filterByColor($request->color);
        }

        if ($request->filled('size')) {
            $query->filterBySize($request->size);
        }

        if ($request->filled('batch_no')) {
            $query->filterByBatchNo($request->batch_no);
        }

        // Get paginated results
        $perPage = $request->get('per_page', 20);
        $stocks = $query->orderBy('created_at', 'desc')->paginate($perPage);

        return response()->json([
            'success' => true,
            'data' => $stocks->items(),
            'pagination' => [
                'current_page' => $stocks->currentPage(),
                'last_page' => $stocks->lastPage(),
                'per_page' => $stocks->perPage(),
                'total' => $stocks->total(),
            ]
        ]);
    }

    /**
     * API endpoint to get filter options for mobile app
     */
    public function apiFilters()
    {
        return response()->json([
            'success' => true,
            'filters' => [
                'names' => Stock::getUniqueNames(),
                'products' => Stock::getUniqueProducts(),
                'styles' => Stock::getUniqueStyles(),
                'colors' => Stock::getUniqueColors(),
                'sizes' => Stock::getUniqueSizes(),
                'batchNos' => Stock::getUniqueBatchNos(),
            ]
        ]);
    }
}
