<?php

namespace App\Http\Controllers;

use App\Models\Task;
use App\Models\User;
use App\Models\Department;
use App\Models\Attendance;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class PerformanceController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('role:Admin');
    }

    /**
     * Display performance analytics dashboard
     */
    public function index(Request $request)
    {
        $dateRange = $this->getDateRange($request);
        
        $data = [
            'overview' => $this->getOverviewStats($dateRange),
            'taskCompletion' => $this->getTaskCompletionStats($dateRange),
            'deadlineAdherence' => $this->getDeadlineAdherenceStats($dateRange),
            'employeePerformance' => $this->getEmployeePerformanceStats($dateRange),
            'departmentPerformance' => $this->getDepartmentPerformanceStats($dateRange),
            'trends' => $this->getTrendData($dateRange),
            'dateRange' => $dateRange
        ];
        
        return view('performance.index', $data);
    }

    /**
     * Get date range from request or default to last 30 days
     */
    private function getDateRange(Request $request)
    {
        $startDate = $request->get('start_date', Carbon::now()->subDays(30)->format('Y-m-d'));
        $endDate = $request->get('end_date', Carbon::now()->format('Y-m-d'));
        
        return [
            'start' => Carbon::parse($startDate),
            'end' => Carbon::parse($endDate)
        ];
    }

    /**
     * Get overview statistics
     */
    private function getOverviewStats($dateRange)
    {
        $totalTasks = Task::whereBetween('created_at', [$dateRange['start'], $dateRange['end']])->count();
        $completedTasks = Task::whereBetween('created_at', [$dateRange['start'], $dateRange['end']])
                             ->where('status', 'completed')
                             ->count();
        $overdueTasks = Task::whereBetween('created_at', [$dateRange['start'], $dateRange['end']])
                           ->where('deadline', '<', now())
                           ->where('status', '!=', 'completed')
                           ->count();
        $onTimeTasks = Task::whereBetween('created_at', [$dateRange['start'], $dateRange['end']])
                          ->where('status', 'completed')
                          ->whereRaw('completed_at <= deadline')
                          ->count();
        
        return [
            'total_tasks' => $totalTasks,
            'completed_tasks' => $completedTasks,
            'completion_rate' => $totalTasks > 0 ? round(($completedTasks / $totalTasks) * 100, 2) : 0,
            'overdue_tasks' => $overdueTasks,
            'on_time_tasks' => $onTimeTasks,
            'on_time_rate' => $completedTasks > 0 ? round(($onTimeTasks / $completedTasks) * 100, 2) : 0
        ];
    }

    /**
     * Get task completion statistics by priority and status
     */
    private function getTaskCompletionStats($dateRange)
    {
        $completionByPriority = Task::whereBetween('created_at', [$dateRange['start'], $dateRange['end']])
                                   ->select('priority', 
                                           DB::raw('COUNT(*) as total'),
                                           DB::raw('SUM(CASE WHEN status = "completed" THEN 1 ELSE 0 END) as completed'))
                                   ->groupBy('priority')
                                   ->get()
                                   ->map(function($item) {
                                       $item->completion_rate = $item->total > 0 ? round(($item->completed / $item->total) * 100, 2) : 0;
                                       return $item;
                                   });
        
        $statusDistribution = Task::whereBetween('created_at', [$dateRange['start'], $dateRange['end']])
                                 ->select('status', DB::raw('COUNT(*) as count'))
                                 ->groupBy('status')
                                 ->get();
        
        return [
            'by_priority' => $completionByPriority,
            'status_distribution' => $statusDistribution
        ];
    }

    /**
     * Get deadline adherence statistics
     */
    private function getDeadlineAdherenceStats($dateRange)
    {
        $completedTasks = Task::whereBetween('created_at', [$dateRange['start'], $dateRange['end']])
                             ->where('status', 'completed')
                             ->whereNotNull('deadline')
                             ->get();
        
        $onTime = $completedTasks->where('completed_at', '<=', function($task) { return $task->deadline; })->count();
        $late = $completedTasks->count() - $onTime;
        
        $avgCompletionTime = $completedTasks->avg(function($task) {
            return Carbon::parse($task->created_at)->diffInDays(Carbon::parse($task->completed_at));
        });
        
        return [
            'total_with_deadline' => $completedTasks->count(),
            'on_time' => $onTime,
            'late' => $late,
            'on_time_percentage' => $completedTasks->count() > 0 ? round(($onTime / $completedTasks->count()) * 100, 2) : 0,
            'avg_completion_days' => round($avgCompletionTime ?? 0, 1)
        ];
    }

    /**
     * Get employee performance statistics
     */
    private function getEmployeePerformanceStats($dateRange)
    {
        return User::role('Staff')
                  ->with('staffProfile.department')
                  ->get()
                  ->map(function($user) use ($dateRange) {
                      $tasks = Task::where('current_assigned_to', $user->id)
                                  ->whereBetween('created_at', [$dateRange['start'], $dateRange['end']])
                                  ->get();
                      
                      $completed = $tasks->where('status', 'completed');
                      $onTime = $completed->filter(function($task) {
                          return $task->deadline && $task->completed_at <= $task->deadline;
                      });
                      
                      $attendance = Attendance::where('user_id', $user->id)
                                            ->whereBetween('date', [$dateRange['start'], $dateRange['end']])
                                            ->where('status', 'present')
                                            ->count();
                      
                      $totalWorkDays = $dateRange['start']->diffInWeekdays($dateRange['end']) + 1;
                      
                      return [
                          'user' => $user,
                          'total_tasks' => $tasks->count(),
                          'completed_tasks' => $completed->count(),
                          'completion_rate' => $tasks->count() > 0 ? round(($completed->count() / $tasks->count()) * 100, 2) : 0,
                          'on_time_tasks' => $onTime->count(),
                          'on_time_rate' => $completed->count() > 0 ? round(($onTime->count() / $completed->count()) * 100, 2) : 0,
                          'attendance_days' => $attendance,
                          'attendance_rate' => $totalWorkDays > 0 ? round(($attendance / $totalWorkDays) * 100, 2) : 0,
                          'department' => $user->staffProfile->department->name ?? 'Unassigned'
                      ];
                  })
                  ->sortByDesc('completion_rate')
                  ->take(10);
    }

    /**
     * Get department performance statistics
     */
    private function getDepartmentPerformanceStats($dateRange)
    {
        return Department::withCount('staffProfiles')
                        ->get()
                        ->map(function($department) use ($dateRange) {
                            $tasks = Task::where('current_department_id', $department->id)
                                        ->whereBetween('created_at', [$dateRange['start'], $dateRange['end']])
                                        ->get();
                            
                            $completed = $tasks->where('status', 'completed');
                            $onTime = $completed->filter(function($task) {
                                return $task->deadline && $task->completed_at <= $task->deadline;
                            });
                            
                            return [
                                'department' => $department,
                                'total_tasks' => $tasks->count(),
                                'completed_tasks' => $completed->count(),
                                'completion_rate' => $tasks->count() > 0 ? round(($completed->count() / $tasks->count()) * 100, 2) : 0,
                                'on_time_rate' => $completed->count() > 0 ? round(($onTime->count() / $completed->count()) * 100, 2) : 0,
                                'staff_count' => $department->staff_profiles_count
                            ];
                        })
                        ->sortByDesc('completion_rate');
    }

    /**
     * Get trend data for charts
     */
    private function getTrendData($dateRange)
    {
        $days = [];
        $current = $dateRange['start']->copy();
        
        while ($current <= $dateRange['end']) {
            $dayTasks = Task::whereDate('created_at', $current)->get();
            $completed = $dayTasks->where('status', 'completed')->count();
            
            $days[] = [
                'date' => $current->format('Y-m-d'),
                'display_date' => $current->format('M j'),
                'total_tasks' => $dayTasks->count(),
                'completed_tasks' => $completed,
                'completion_rate' => $dayTasks->count() > 0 ? round(($completed / $dayTasks->count()) * 100, 2) : 0
            ];
            
            $current->addDay();
        }
        
        return $days;
    }

    /**
     * Export performance data as CSV
     */
    public function export(Request $request)
    {
        $dateRange = $this->getDateRange($request);
        $employeePerformance = $this->getEmployeePerformanceStats($dateRange);
        
        $filename = 'performance_report_' . $dateRange['start']->format('Y-m-d') . '_to_' . $dateRange['end']->format('Y-m-d') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];
        
        $callback = function() use ($employeePerformance) {
            $file = fopen('php://output', 'w');
            
            // CSV Headers
            fputcsv($file, [
                'Employee Name',
                'Department',
                'Total Tasks',
                'Completed Tasks',
                'Completion Rate (%)',
                'On-Time Tasks',
                'On-Time Rate (%)',
                'Attendance Days',
                'Attendance Rate (%)'
            ]);
            
            // Data rows
            foreach ($employeePerformance as $performance) {
                fputcsv($file, [
                    $performance['user']->name,
                    $performance['department'],
                    $performance['total_tasks'],
                    $performance['completed_tasks'],
                    $performance['completion_rate'],
                    $performance['on_time_tasks'],
                    $performance['on_time_rate'],
                    $performance['attendance_days'],
                    $performance['attendance_rate']
                ]);
            }
            
            fclose($file);
        };
        
        return response()->stream($callback, 200, $headers);
    }
}