<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Payroll;
use App\Models\User;
use App\Models\Attendance;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Models\LeaveRequest;
use App\Models\Reimbursement;

class PayrollController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        
        // Get individual payrolls
        $individualQuery = Payroll::with(['user', 'user.staffProfile'])
                                 ->where('payroll_type', 'individual');
        
        // Get bulk payrolls grouped by batch_id
        $bulkQuery = Payroll::select('batch_id', 'year', 'month', 'payroll_type')
                           ->selectRaw('COUNT(*) as employee_count')
                           ->selectRaw('SUM(total_payout) as total_amount')
                           ->where('payroll_type', 'bulk')
                           ->whereNotNull('batch_id')
                           ->groupBy('batch_id', 'year', 'month', 'payroll_type');
        
        // Filter by month/year if provided
        if ($request->filled('month')) {
            $individualQuery->where('month', $request->month);
            $bulkQuery->where('month', $request->month);
        }
        if ($request->filled('year')) {
            $individualQuery->where('year', $request->year);
            $bulkQuery->where('year', $request->year);
        }
        
        // Filter by user if admin, otherwise show only current user's payroll
        if ($user->hasRole('Admin')) {
            if ($request->filled('user_id')) {
                $individualQuery->where('user_id', $request->user_id);
                // For bulk payrolls, we don't filter by user_id as they represent groups
            }
            $users = User::where('email', '!=', 'admin@example.com')->get();
        } else {
            $individualQuery->where('user_id', $user->id);
            // Staff users can't see bulk payrolls in the main list
            $bulkQuery->whereRaw('1 = 0'); // This will return no results
            $users = collect();
        }
        
        $individualPayrolls = $individualQuery->latest('year')->latest('month')->get();
        $bulkPayrolls = $bulkQuery->latest('year')->latest('month')->get();
        
        // Combine and paginate manually
        $allPayrolls = $individualPayrolls->concat($bulkPayrolls)
                                        ->sortByDesc(function($item) {
                                            return $item->year * 100 + $item->month;
                                        });
        
        // Simple pagination
        $perPage = 15;
        $currentPage = $request->get('page', 1);
        $offset = ($currentPage - 1) * $perPage;
        $paginatedPayrolls = $allPayrolls->slice($offset, $perPage);
        
        $payrolls = new \Illuminate\Pagination\LengthAwarePaginator(
            $paginatedPayrolls,
            $allPayrolls->count(),
            $perPage,
            $currentPage,
            ['path' => $request->url(), 'query' => $request->query()]
        );
        
        return view('payroll.index', compact('payrolls', 'users'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $this->authorize('create', Payroll::class);
        
        $users = User::where('email', '!=', 'admin@example.com')->with('staffProfile')->get();
        
        return view('payroll.create', compact('users'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //return $request;
        Log::info('Store method called', [
            'request_data' => $request->all(),
            'method' => $request->method(),
            'url' => $request->url()
        ]);
        
        Log::info('Raw payroll_type from request', ['payroll_type' => $request->input('payroll_type')]);
        
        try {
            $this->authorize('create', Payroll::class);
            Log::info('Authorization passed for payroll creation');
        } catch (\Exception $e) {
            Log::error('Authorization failed for payroll creation', ['error' => $e->getMessage()]);
            throw $e;
        }
        
        try {
            $validated = $request->validate([
                 'payroll_type' => 'required|in:individual,bulk',
                 'user_id' => 'required_if:payroll_type,individual|nullable|exists:users,id',
                 'month' => 'required|integer|min:1|max:12',
                 'year' => 'required|integer|min:2020|max:2030',
                 'special_deduction' => 'nullable|numeric|min:0',
                 'special_deduction_note' => 'nullable|string|required_with:special_deduction',
             ]);
            Log::info('Validation passed', ['validated_data' => $validated]);
        } catch (\Exception $e) {
            Log::error('Validation failed', ['error' => $e->getMessage(), 'request_data' => $request->all()]);
            throw $e;
        }
        
        Log::info('Payroll type detected', ['payroll_type' => $validated['payroll_type']]);
        
        Log::info('Payroll creation started', [
            'payroll_type' => $validated['payroll_type'],
            'month' => $validated['month'],
            'year' => $validated['year']
        ]);
        
        if ($validated['payroll_type'] === 'individual') {
            Log::info('Creating individual payroll');
            return $this->createIndividualPayroll($validated);
        } elseif ($validated['payroll_type'] === 'bulk') {
            Log::info('Creating bulk payroll');
            return $this->createBulkPayroll($validated);
        } else {
            Log::error('Invalid payroll type', ['type' => $validated['payroll_type']]);
            return redirect()->route('payrolls.index')
                            ->with('error', 'Invalid payroll type selected.');
        }
    }
    
    /**
     * Create payroll for a single user
     */
    private function createIndividualPayroll($validated)
    {
        $user = User::with('staffProfile')->find($validated['user_id']);
        
        // Check if payroll already exists for this user, month, and year
        $existingPayroll = Payroll::where('user_id', $validated['user_id'])
                                 ->where('month', $validated['month'])
                                 ->where('year', $validated['year'])
                                 ->first();
        
        if ($existingPayroll) {
            return back()->withErrors(['user_id' => 'Payroll for this user in this month/year already exists.']);
        }
        
        $salaryDetails = $this->getSalaryDetails($user->staffProfile);
        
        $payroll = $this->calculateAndCreatePayroll(
            $user->id,
            $validated['month'],
            $validated['year'],
            $salaryDetails,
            'individual',
            null,
            $validated
        );
        
        return redirect()->route('payrolls.index')
                        ->with('success', 'Individual payroll created successfully.');
    }
    
    /**
     * Helper to extract salary details from profile
     */
    private function getSalaryDetails($profile)
    {
        if (!$profile) {
            return ['monthly_compensation' => 0];
        }

        $basic = $profile->basic_salary ?? 0;
        $hra = $profile->hra ?? 0;
        $conveyance = $profile->conveyance ?? 0;
        $special = $profile->special_allowance ?? 0;
        
        // If granular fields are empty but legacy salary exists, fallback (optional, but good for safety)
        if ($basic == 0 && $profile->salary > 0) {
            // Assume legacy mode: salary is gross/net
            return [
                'monthly_compensation' => $profile->salary,
                'gross_salary' => $profile->salary,
                'net_salary' => $profile->salary
            ];
        }

        $gross = $basic + $hra + $conveyance + $special;
        
        $employerPf = $profile->employer_pf ?? 0;
        $employerEsi = $profile->employer_esi ?? 0;
        
        $ctc = $gross + $employerPf + $employerEsi;
        
        $employeePf = $profile->employee_pf ?? 0;
        $employeeEsi = $profile->employee_esi ?? 0;
        $profTax = $profile->professional_tax ?? 0;
        
        $deductions = $employeePf + $employeeEsi + $profTax;
        $netSalary = $gross - $deductions;

        return [
            'basic_salary' => $basic,
            'hra' => $hra,
            'conveyance' => $conveyance,
            'special_allowance' => $special,
            'gross_salary' => $gross,
            'employer_pf' => $employerPf,
            'employer_esi' => $employerEsi,
            'ctc' => $ctc,
            'employee_pf' => $employeePf,
            'employee_esi' => $employeeEsi,
            'professional_tax' => $profTax,
            'net_salary' => $netSalary,
            'monthly_compensation' => $netSalary // Used for per-day calculation as per requirement
        ];
    }
    
    /**
     * Create payroll for all staff members
     */
    private function createBulkPayroll($validated)
    {
        try {
            // Get all users except admin
            $staff = User::where('email', '!=', 'admin@example.com')->with('staffProfile')->get();
            //return $staff;
            if ($staff->isEmpty()) {
                return redirect()->route('payrolls.index')
                                ->with('error', 'No staff members found to create payroll for.');
            }
            
            $generated = 0;
            $skipped = 0;
            $errors = [];
            
            // Generate unique batch ID for this bulk payroll
            $batchId = 'BULK_' . $validated['year'] . '_' . str_pad($validated['month'], 2, '0', STR_PAD_LEFT) . '_' . uniqid();
            
            foreach ($staff as $user) {
                try {
                    // Check if payroll already exists for this user and period
                    $existingPayroll = Payroll::where('user_id', $user->id)
                                             ->where('month', $validated['month'])
                                             ->where('year', $validated['year'])
                                             ->first();
                    
                    if ($existingPayroll) {
                        $skipped++;
                        continue;
                    }
                    
                    // Get monthly compensation from staff profile
                    if (!$user->staffProfile) {
                        $errors[] = "Staff member {$user->name} has no profile";
                        continue;
                    }
                    
                    $salaryDetails = $this->getSalaryDetails($user->staffProfile);
                    
                    // Create payroll record
                    $this->createPayrollRecord(
                        $user->id,
                        $validated['month'],
                        $validated['year'],
                        $salaryDetails,
                        'bulk',
                        $batchId
                    );
                    
                    $generated++;
                    
                } catch (\Exception $e) {
                    $errors[] = "Failed to create payroll for {$user->name}: " . $e->getMessage();
                }
            }
            
            // Prepare success message
            $message = "Bulk payroll created successfully! Generated payroll for {$generated} staff members.";
            if ($skipped > 0) {
                $message .= " Skipped {$skipped} existing records.";
            }
            if (!empty($errors)) {
                $message .= " Errors: " . implode(', ', $errors);
            }
            
            return redirect()->route('payrolls.index')
                            ->with('success', $message);
                            
        } catch (\Exception $e) {
            Log::error('Bulk payroll creation failed: ' . $e->getMessage());
            return redirect()->route('payrolls.index')
                            ->with('error', 'Failed to create bulk payroll: ' . $e->getMessage());
        }
    }
    
    /**
     * Create payroll record with attendance calculation
     */
    private function createPayrollRecord($userId, $month, $year, $salaryData, $payrollType = 'individual', $batchId = null)
    {
        // Get attendance data for the month
        $startDate = Carbon::create($year, $month, 1);
        $endDate = $startDate->copy()->endOfMonth();
        
        // Count total working days (Monday to Saturday, excluding Sundays)
        $totalWorkingDays = 0;
        $currentDate = $startDate->copy();
        while ($currentDate <= $endDate) {
            if ($currentDate->dayOfWeek !== Carbon::SUNDAY) {
                $totalWorkingDays++;
            }
            $currentDate->addDay();
        }
        
        // Count days present (present, late, half_day)
        $daysPresent = Attendance::where('user_id', $userId)
                                ->whereBetween('date', [$startDate, $endDate])
                                ->whereIn('status', ['present', 'late', 'half_day'])
                                ->count();
        
        // Calculate Paid Leave Days
        $paidLeaveDays = 0;
        $approvedLeaves = LeaveRequest::with('leaveType')
                                    ->where('user_id', $userId)
                                    ->where('status', 'approved')
                                    ->where(function($query) use ($startDate, $endDate) {
                                        $query->whereBetween('start_date', [$startDate, $endDate])
                                            ->orWhereBetween('end_date', [$startDate, $endDate]);
                                    })
                                    ->get();

        foreach ($approvedLeaves as $leave) {
            // Determine intersection of leave days with the current month
            $leaveStart = Carbon::parse($leave->start_date);
            $leaveEnd = Carbon::parse($leave->end_date);
            
            // Clamp dates to the current month
            $effectiveStart = $leaveStart->max($startDate);
            $effectiveEnd = $leaveEnd->min($endDate);
            
            // Calculate working days in this intersection
            $days = Attendance::getWorkingDaysBetween($effectiveStart, $effectiveEnd);
            
            if ($leave->is_half_day) {
                $days = 0.5; // Override for half-day requests
            }
            
            // Calculate paid portion based on deduction percentage
            // 0% deduction = 100% paid
            // 100% deduction = 0% paid
            // 50% deduction = 50% paid
            $paidRatio = 1 - ($leave->leaveType->deduction_percentage / 100);
            
            $paidLeaveDays += $days * $paidRatio;
        }

        $daysAbsent = $totalWorkingDays - $daysPresent - $paidLeaveDays;
        // Ensure days absent is not negative (in case of overlaps or manual adjustments)
        $daysAbsent = max(0, $daysAbsent);
        
        // Fetch Reimbursements
        $reimbursements = Reimbursement::where('user_id', $userId)
            ->where('status', 'approved')
            ->whereNull('payroll_id')
            ->get();

        $reimbursementTotal = $reimbursements->sum('approved_amount');

        // Special Deduction
        $specialDeduction = $extraData['special_deduction'] ?? 0;
        $specialDeductionNote = $extraData['special_deduction_note'] ?? null;

        // Prepare payroll data
        $data = [
            'user_id' => $userId,
            'payroll_type' => $payrollType,
            'batch_id' => $batchId,
            'month' => $month,
            'year' => $year,
            'days_present' => $daysPresent,
            'paid_leave_days' => $paidLeaveDays,
            'days_absent' => $daysAbsent,
            'total_work_days' => $totalWorkingDays,
            'reimbursement_total' => $reimbursementTotal,
            'special_deduction' => $specialDeduction,
            'special_deduction_note' => $specialDeductionNote,
            'daily_rate' => 0, // Will be calculated by the model
            'total_payout' => 0, // Will be calculated by the model
        ];

        // Merge salary details
        if (is_array($salaryData)) {
            $data = array_merge($data, $salaryData);
        } else {
            // Legacy support if scalar passed
            $data['monthly_compensation'] = $salaryData;
        }

        // Create payroll record
        $payroll = Payroll::create($data);

        // Link Reimbursements
        if ($reimbursements->count() > 0) {
            Reimbursement::whereIn('id', $reimbursements->pluck('id'))
                ->update(['payroll_id' => $payroll->id, 'status' => 'paid']);
        }
        
        // Calculate payout using the model method
        $payroll->calculatePayout();
        $payroll->save();
        
        return $payroll;
    }
    
    /**
     * Calculate attendance and create payroll record (legacy method for individual payrolls)
     */
    private function calculateAndCreatePayroll($userId, $month, $year, $monthlyCompensation, $payrollType = 'individual', $batchId = null, $extraData = [])
    {
        return $this->createPayrollRecord($userId, $month, $year, $monthlyCompensation, $payrollType, $batchId, $extraData);
    }

    /**
     * Display the specified resource.
     */
    public function show(Payroll $payroll)
    {
        $this->authorize('view', $payroll);
        
        $payroll->load('user.staffProfile.department');
        
        return view('payroll.show', compact('payroll'));
    }

    /**
     * Download the payroll as PDF.
     */
    public function downloadPdf(Payroll $payroll)
    {
        $this->authorize('view', $payroll);
        
        $payroll->load('user.staffProfile.department');
        
        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('payroll.pdf', compact('payroll'));
        
        return $pdf->download("payslip_{$payroll->user->id}_{$payroll->month}_{$payroll->year}.pdf");
    }
    
    /**
     * Display bulk payroll details
     */
    public function showBulk($batchId)
    {
        $this->authorize('viewAny', Payroll::class);
        
        $payrolls = Payroll::with(['user', 'user.staffProfile.department'])
                          ->where('batch_id', $batchId)
                          ->where('payroll_type', 'bulk')
                          ->orderBy('user_id')
                          ->get();
        
        if ($payrolls->isEmpty()) {
            abort(404, 'Bulk payroll not found.');
        }
        
        $batchInfo = [
            'batch_id' => $batchId,
            'year' => $payrolls->first()->year,
            'month' => $payrolls->first()->month,
            'employee_count' => $payrolls->count(),
            'total_amount' => $payrolls->sum('total_payout'),
            'created_at' => $payrolls->first()->created_at
        ];
        
        return view('payroll.bulk-show', compact('payrolls', 'batchInfo'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Payroll $payroll)
    {
        $this->authorize('update', $payroll);
        
        $users = User::where('email', '!=', 'admin@example.com')->with('staffProfile')->get();
        
        return view('payroll.edit', compact('payroll', 'users'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Payroll $payroll)
    {
        $this->authorize('update', $payroll);
        
        $validated = $request->validate([
            'user_id' => 'required|exists:users,id',
            'month' => 'required|integer|min:1|max:12',
            'year' => 'required|integer|min:2020|max:2030',
            // Detailed salary structure
            'basic_salary' => 'nullable|numeric|min:0',
            'hra' => 'nullable|numeric|min:0',
            'conveyance' => 'nullable|numeric|min:0',
            'special_allowance' => 'nullable|numeric|min:0',
            'employer_pf' => 'nullable|numeric|min:0',
            'employer_esi' => 'nullable|numeric|min:0',
            'employee_pf' => 'nullable|numeric|min:0',
            'employee_esi' => 'nullable|numeric|min:0',
            'professional_tax' => 'nullable|numeric|min:0',
        ]);
        
        // Check if payroll already exists for this user, month, and year (excluding current record)
        $existingPayroll = Payroll::where('user_id', $validated['user_id'])
                                 ->where('month', $validated['month'])
                                 ->where('year', $validated['year'])
                                 ->where('id', '!=', $payroll->id)
                                 ->first();
        
        if ($existingPayroll) {
            return back()->withErrors(['month' => 'Payroll for this user in this month/year already exists.']);
        }
        
        // Get attendance data for the month
        $startDate = Carbon::create($validated['year'], $validated['month'], 1);
        $endDate = $startDate->copy()->endOfMonth();
        
        // Count total working days (Monday to Saturday)
        $totalWorkingDays = 0;
        $currentDate = $startDate->copy();
        while ($currentDate <= $endDate) {
            if ($currentDate->dayOfWeek !== Carbon::SUNDAY) {
                $totalWorkingDays++;
            }
            $currentDate->addDay();
        }
        
        // Count days present (present, late, half_day)
        $daysPresent = Attendance::where('user_id', $validated['user_id'])
                                ->whereBetween('date', [$startDate, $endDate])
                                ->whereIn('status', ['present', 'late', 'half_day'])
                                ->count();
        
        // Calculate Paid Leave Days
        $paidLeaveDays = 0;
        $approvedLeaves = LeaveRequest::with('leaveType')
                                    ->where('user_id', $validated['user_id'])
                                    ->where('status', 'approved')
                                    ->where(function($query) use ($startDate, $endDate) {
                                        $query->whereBetween('start_date', [$startDate, $endDate])
                                            ->orWhereBetween('end_date', [$startDate, $endDate]);
                                    })
                                    ->get();

        foreach ($approvedLeaves as $leave) {
            $leaveStart = Carbon::parse($leave->start_date);
            $leaveEnd = Carbon::parse($leave->end_date);
            $effectiveStart = $leaveStart->max($startDate);
            $effectiveEnd = $leaveEnd->min($endDate);
            $days = Attendance::getWorkingDaysBetween($effectiveStart, $effectiveEnd);
            
            if ($leave->is_half_day) {
                $days = 0.5;
            }
            
            $paidRatio = 1 - ($leave->leaveType->deduction_percentage / 100);
            $paidLeaveDays += $days * $paidRatio;
        }

        $daysAbsent = $totalWorkingDays - $daysPresent - $paidLeaveDays;
        $daysAbsent = max(0, $daysAbsent);
        
        // Calculate totals
        $basic = $request->input('basic_salary', 0);
        $hra = $request->input('hra', 0);
        $conveyance = $request->input('conveyance', 0);
        $special = $request->input('special_allowance', 0);
        $gross = $basic + $hra + $conveyance + $special;
        
        $empPf = $request->input('employer_pf', 0);
        $empEsi = $request->input('employer_esi', 0);
        $ctc = $gross + $empPf + $empEsi;
        
        $dedPf = $request->input('employee_pf', 0);
        $dedEsi = $request->input('employee_esi', 0);
        $profTax = $request->input('professional_tax', 0);
        $deductions = $dedPf + $dedEsi + $profTax;
        
        $net = $gross - $deductions;

        $payroll->update([
            'user_id' => $validated['user_id'],
            'month' => $validated['month'],
            'year' => $validated['year'],
            'days_present' => $daysPresent,
            'paid_leave_days' => $paidLeaveDays,
            'days_absent' => $daysAbsent,
            'total_work_days' => $totalWorkingDays,
            'basic_salary' => $basic,
            'hra' => $hra,
            'conveyance' => $conveyance,
            'special_allowance' => $special,
            'gross_salary' => $gross,
            'employer_pf' => $empPf,
            'employer_esi' => $empEsi,
            'ctc' => $ctc,
            'employee_pf' => $dedPf,
            'employee_esi' => $dedEsi,
            'professional_tax' => $profTax,
            'net_salary' => $net,
            'monthly_compensation' => $net, // Per user requirement for per-day calculation basis
            'daily_rate' => 0, // Will be calculated by the model
            'total_payout' => 0, // Will be calculated by the model
        ]);
        
        // Calculate payout using the model method
        $payroll->calculatePayout();
        $payroll->save();
        
        return redirect()->route('payrolls.show', $payroll)
                        ->with('success', 'Payroll updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Payroll $payroll)
    {
        $this->authorize('delete', $payroll);
        
        $payroll->delete();
        
        return redirect()->route('payrolls.index')
                        ->with('success', 'Payroll deleted successfully.');
    }

    /**
     * Remove an entire batch of payrolls from storage.
     */
    public function destroyBatch($batchId)
    {
        $this->authorize('viewAny', Payroll::class);
        
        // Find all payrolls in this batch
        $payrolls = Payroll::where('batch_id', $batchId)
                          ->where('payroll_type', 'bulk')
                          ->get();
        
        if ($payrolls->isEmpty()) {
            return redirect()->route('payrolls.index')
                            ->with('error', 'Batch payroll not found.');
        }
        
        // Check authorization for each payroll in the batch
        foreach ($payrolls as $payroll) {
            $this->authorize('delete', $payroll);
        }
        
        $employeeCount = $payrolls->count();
        $batchInfo = $payrolls->first();
        $monthYear = date('F Y', mktime(0, 0, 0, $batchInfo->month, 1, $batchInfo->year));
        
        // Delete all payrolls in the batch
        Payroll::where('batch_id', $batchId)
               ->where('payroll_type', 'bulk')
               ->delete();
        
        return redirect()->route('payrolls.index')
                        ->with('success', "Bulk payroll for {$monthYear} deleted successfully. Removed {$employeeCount} payroll records.");
    }
    

    
    // Note: Approve and markPaid methods removed as the simplified payroll structure
    // focuses on attendance-based calculations without status tracking
    
    /**
     * Generate payroll report
     */
    public function report(Request $request)
    {
        $this->authorize('viewAny', Payroll::class);
        
        $validated = $request->validate([
            'month' => 'required|integer|min:1|max:12',
            'year' => 'required|integer|min:2020|max:2030',
        ]);
        
        $payrolls = Payroll::with(['user', 'user.staffProfile'])
                          ->where('month', $validated['month'])
                          ->where('year', $validated['year'])
                          ->get();
        
        // Calculate summary statistics
        $summary = [
            'total_employees' => $payrolls->count(),
            'total_monthly_compensation' => $payrolls->sum('monthly_compensation'),
            'total_payout' => $payrolls->sum('total_payout'),
            'total_days_present' => $payrolls->sum('days_present'),
            'total_days_absent' => $payrolls->sum('days_absent'),
            'total_work_days' => $payrolls->sum('total_work_days'),
        ];
        
        return view('payroll.report', compact('payrolls', 'summary', 'validated'));
    }

    // API Methods for Mobile App
    
    /**
     * API: Get payroll records
     */
    public function apiIndex(Request $request)
    {
        $user = $request->user();
        
        if ($user->hasRole('Admin')) {
            // Get individual payrolls
            $individualQuery = Payroll::with(['user', 'user.staffProfile'])
                                     ->where('payroll_type', 'individual');
            
            // Get bulk payrolls grouped by batch_id
            $bulkQuery = Payroll::select('batch_id', 'year', 'month', 'payroll_type')
                               ->selectRaw('COUNT(*) as employee_count')
                               ->selectRaw('SUM(total_payout) as total_amount')
                               ->where('payroll_type', 'bulk')
                               ->whereNotNull('batch_id')
                               ->groupBy('batch_id', 'year', 'month', 'payroll_type');
            
            // Filter by month/year if provided
            if ($request->filled('month')) {
                $individualQuery->where('month', $request->month);
                $bulkQuery->where('month', $request->month);
            }
            if ($request->filled('year')) {
                $individualQuery->where('year', $request->year);
                $bulkQuery->where('year', $request->year);
            }
            
            // Filter by user if provided
            if ($request->filled('user_id')) {
                $individualQuery->where('user_id', $request->user_id);
            }
            
            $individualPayrolls = $individualQuery->latest()->paginate(10);
            $bulkPayrolls = $bulkQuery->latest()->paginate(10);
            
            return response()->json([
                'individual_payrolls' => $individualPayrolls,
                'bulk_payrolls' => $bulkPayrolls
            ]);
        } else {
            // Staff can only see their own payroll
            $payrolls = Payroll::where('user_id', $user->id)
                              ->with(['user', 'user.staffProfile']);
            
            if ($request->filled('month')) {
                $payrolls->where('month', $request->month);
            }
            if ($request->filled('year')) {
                $payrolls->where('year', $request->year);
            }
            
            $payrolls = $payrolls->latest()->paginate(15);
            
            return response()->json($payrolls);
        }
    }
    
    /**
     * API: Create payroll record
     */
    public function apiStore(Request $request)
    {
        $user = $request->user();
        
        // Only admin can create payroll
        if (!$user->hasRole('Admin')) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }
        
        $validated = $request->validate([
            'user_id' => 'required|exists:users,id',
            'year' => 'required|integer|min:2020|max:2030',
            'month' => 'required|integer|min:1|max:12',
            'days_present' => 'required|integer|min:0',
            'days_absent' => 'required|integer|min:0',
            'total_work_days' => 'required|integer|min:1',
            'monthly_compensation' => 'required|numeric|min:0',
            'daily_rate' => 'required|numeric|min:0',
            'total_payout' => 'required|numeric|min:0',
            'payroll_type' => 'required|in:individual,bulk',
            'batch_id' => 'nullable|string',
        ]);
        
        // Check if payroll already exists for this user and month
        $existingPayroll = Payroll::where('user_id', $validated['user_id'])
                                 ->where('year', $validated['year'])
                                 ->where('month', $validated['month'])
                                 ->first();
        
        if ($existingPayroll) {
            return response()->json(['error' => 'Payroll already exists for this user and month.'], 422);
        }
        
        $payroll = Payroll::create($validated);
        
        return response()->json([
            'message' => 'Payroll created successfully.',
            'payroll' => $payroll->load(['user', 'user.staffProfile'])
        ], 201);
    }
    
    /**
     * API: Get specific payroll record
     */
    public function apiShow(Request $request, $id)
    {
        $user = $request->user();
        $payroll = Payroll::with(['user', 'user.staffProfile'])->findOrFail($id);
        
        // Staff can only view their own payroll
        if (!$user->hasRole('Admin') && $payroll->user_id !== $user->id) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }
        
        return response()->json($payroll);
    }
    
    /**
     * API: Update payroll record
     */
    public function apiUpdate(Request $request, $id)
    {
        $user = $request->user();
        
        // Only admin can update payroll
        if (!$user->hasRole('Admin')) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }
        
        $payroll = Payroll::findOrFail($id);
        
        $validated = $request->validate([
            'days_present' => 'sometimes|integer|min:0',
            'days_absent' => 'sometimes|integer|min:0',
            'total_work_days' => 'sometimes|integer|min:1',
            'monthly_compensation' => 'sometimes|numeric|min:0',
            'daily_rate' => 'sometimes|numeric|min:0',
            'total_payout' => 'sometimes|numeric|min:0',
        ]);
        
        $payroll->update($validated);
        
        return response()->json([
            'message' => 'Payroll updated successfully.',
            'payroll' => $payroll->fresh()->load(['user', 'user.staffProfile'])
        ]);
    }
    
    /**
     * API: Delete payroll record
     */
    public function apiDestroy(Request $request, $id)
    {
        $user = $request->user();
        
        // Only admin can delete payroll
        if (!$user->hasRole('Admin')) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }
        
        $payroll = Payroll::findOrFail($id);
        $payroll->delete();
        
        return response()->json(['message' => 'Payroll deleted successfully.']);
    }
    
    /**
     * API: Bulk create payrolls
     */
    public function apiBulkCreate(Request $request)
    {
        $user = $request->user();
        
        // Only admin can create bulk payroll
        if (!$user->hasRole('Admin')) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }
        
        $validated = $request->validate([
            'year' => 'required|integer|min:2020|max:2030',
            'month' => 'required|integer|min:1|max:12',
            'department_id' => 'nullable|exists:departments,id',
        ]);
        
        $batchId = 'BULK_' . $validated['year'] . '_' . str_pad($validated['month'], 2, '0', STR_PAD_LEFT) . '_' . time();
        
        // Get staff users
        $staffQuery = User::role('Staff')->with('staffProfile');
        
        if ($validated['department_id']) {
            $staffQuery->whereHas('staffProfile', function($q) use ($validated) {
                $q->where('department_id', $validated['department_id']);
            });
        }
        
        $staffUsers = $staffQuery->get();
        $createdPayrolls = [];
        
        foreach ($staffUsers as $staff) {
            // Check if payroll already exists
            $existingPayroll = Payroll::where('user_id', $staff->id)
                                     ->where('year', $validated['year'])
                                     ->where('month', $validated['month'])
                                     ->first();
            
            if ($existingPayroll) {
                continue; // Skip if already exists
            }
            
            // Calculate attendance data
            $attendances = Attendance::where('user_id', $staff->id)
                                   ->whereYear('date', $validated['year'])
                                   ->whereMonth('date', $validated['month'])
                                   ->get();
            
            $daysPresent = $attendances->whereIn('status', ['present', 'late', 'half_day'])->count();
            $daysAbsent = $attendances->where('status', 'absent')->count();
            $totalWorkDays = Attendance::getWorkingDaysInMonth($validated['year'], $validated['month']);
            
            $monthlyCompensation = $staff->staffProfile->monthly_compensation ?? 0;
            $dailyRate = $totalWorkDays > 0 ? $monthlyCompensation / $totalWorkDays : 0;
            $totalPayout = $daysPresent * $dailyRate;
            
            $payroll = Payroll::create([
                'user_id' => $staff->id,
                'year' => $validated['year'],
                'month' => $validated['month'],
                'days_present' => $daysPresent,
                'days_absent' => $daysAbsent,
                'total_work_days' => $totalWorkDays,
                'monthly_compensation' => $monthlyCompensation,
                'daily_rate' => $dailyRate,
                'total_payout' => $totalPayout,
                'payroll_type' => 'bulk',
                'batch_id' => $batchId,
            ]);
            
            $createdPayrolls[] = $payroll;
        }
        
        return response()->json([
            'message' => 'Bulk payroll created successfully.',
            'batch_id' => $batchId,
            'created_count' => count($createdPayrolls),
            'payrolls' => $createdPayrolls
        ], 201);
    }
    
    /**
     * API: Approve payroll
     */
    public function apiApprove(Request $request, $id)
    {
        $user = $request->user();
        
        // Only admin can approve payroll
        if (!$user->hasRole('Admin')) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }
        
        $payroll = Payroll::findOrFail($id);
        
        // Add approval logic here if you have approval status field
        // For now, just return success
        
        return response()->json([
            'message' => 'Payroll approved successfully.',
            'payroll' => $payroll->load(['user', 'user.staffProfile'])
        ]);
    }
    
    /**
     * API: Mark payroll as paid
     */
    public function apiMarkPaid(Request $request, $id)
    {
        $user = $request->user();
        
        // Only admin can mark as paid
        if (!$user->hasRole('Admin')) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }
        
        $payroll = Payroll::findOrFail($id);
        
        // Add paid status logic here if you have paid status field
        // For now, just return success
        
        return response()->json([
            'message' => 'Payroll marked as paid successfully.',
            'payroll' => $payroll->load(['user', 'user.staffProfile'])
        ]);
    }
}
