<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Task;

class UpdateTaskPriorities extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'tasks:update-priorities';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Update task priorities based on deadline proximity and mark overdue tasks';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Starting task priority updates...');
        
        // Get all active tasks with deadlines
        $tasks = Task::whereIn('status', ['pending', 'in_progress'])
                    ->whereNotNull('deadline')
                    ->get();
        
        $updatedCount = 0;
        $overdueCount = 0;
        
        foreach ($tasks as $task) {
            $oldPriority = $task->priority;
            $oldOverdue = $task->is_overdue;
            
            // Escalate priority based on deadline
            $task->escalatePriorityByDeadline();
            
            // Refresh to get updated values
            $task->refresh();
            
            if ($oldPriority !== $task->priority || $oldOverdue !== $task->is_overdue) {
                $updatedCount++;
                
                if ($task->is_overdue && !$oldOverdue) {
                    $overdueCount++;
                    $this->warn("Task '{$task->title}' is now overdue!");
                } elseif ($oldPriority !== $task->priority) {
                    $this->info("Task '{$task->title}' priority updated from {$oldPriority} to {$task->priority}");
                }
            }
        }
        
        $this->info("Task priority update completed!");
        $this->info("Total tasks processed: {$tasks->count()}");
        $this->info("Tasks updated: {$updatedCount}");
        $this->info("New overdue tasks: {$overdueCount}");
        
        return 0;
    }
}
